'-----------------------------------------------------------[ MODULE ]-----------------------------------------------------------
Rem
	bbdoc:	ChaosConsole
	abbout:  ChaosConsole offers a quake-like console, including access to user defined types and functions
End Rem
Module chaos.console
ModuleInfo "Version: 0.3-rel_1"
ModuleInfo "Author: hamZta"
ModuleInfo "License: Public Domain"
ModuleInfo "Contact: hamZta@chaos-interactive.de"
ModuleInfo "Homepage: www.chaos-interactive.de"

ModuleInfo "History: 0.1-rel_1 Erster Release (19.07.2009)"
ModuleInfo "History: 0.3-rel_1 Umstellung des Parsers auf Lua (19.07.2009)"

'-----------------------------------------------------------[ INIT ]--------------------------------------------------------------
SuperStrict

Import BRL.LinkedList
Import BRL.Map
Import BRL.Max2D
Import BRL.Reflection
Import Pub.Lua

Const CHAOSCONSOLE_VERSION:String = "0.3-rel_1"

'-----------------------------------------------------------[ PRIVATE ]-----------------------------------------------------------
Private

Include "console_lua_aux.bmx"
Include "wrapper.bmx"

Const DKEY_LEFT:Int = 0
Const DKEY_RIGHT:Int = 1
Const DKEY_UP:Int = 2
Const DKEY_DOWN:Int = 3
Const DKEY_DEL:Int = 4

'-----------------------------------------------------------[ PUBLIC ]------------------------------------------------------------
Public

Type TChaosConsole
	
	Global _instances:TList = New TList

	Field _objWrapper:TObjectWrapper
	Field _lineBuffer:String[100]
	Field _linePtr:Int
	Field _startLine:Int
	Field _lineDisplayCount:Int
	Field _charsPerLine:Int
	
	Field _inputLine:String
	Field _inputHistory:String[], _historyIndex:Int
	
	Field _y:Int
	Field _w:Int
	Field _h:Int
	
	Field _colorBR:Byte, _colorBG:Byte, _colorBB:Byte
	Field _colorFR:Byte, _colorFG:Byte, _colorFB:Byte
	Field _backImage:TImage, _alpha:Float
	
	Field _show:Byte
	Field _cursorBlinkTimer:Int
	Field _cursorShow:Int
	Field _cursorPos:Int
	
	Field _fontHeight:Int
	Field _font:TImageFont
	
	Field _keyRepeatTime:Int[5], _keyRepeatTimer:Int
	
	Field _funcMap:TMap
	
	Field _fadeTimer:Int, _fadeDir:Int

	Field _luaState:Byte Ptr
	Field _customLuaState:Byte
	
	Rem
		bbdoc: Returns an instance of TChaosConsole utilizing the given Lua state.
		returns: TChaosConsole
		about:
	End Rem
	Function getPerLuaState:TChaosConsole(ls:Byte Ptr)
		For Local inst:TChaosConsole = EachIn TChaosConsole._instances
			If inst._luaState = ls Return inst
		Next
	End Function
	'---------------------------------------------------------------------------------------------------------------------------------
	
	Method New()
		_objWrapper = New TObjectWrapper
		_startLine = 0
		_funcMap = New TMap
		_colorBR = 0
		_colorBG = 0
		_colorBB = 0
		_colorFR = 255
		_colorFG = 255
		_colorFB = 255
		
		setFont(Null)		
		setDimensions(GraphicsWidth(), GraphicsHeight() / 3)

		output("~~ ChaosConsole ~~")
		output("Version: " + CHAOSCONSOLE_VERSION)
		output("Lua: " + LUA_RELEASE)
		output(LUA_COPYRIGHT)
		output("-----------------------------------------------------------")
		registerObject("console", Self)

		_luaState = luaL_newstate()
		luaL_openlibs(_luaState)
		initLua()

		TChaosConsole._instances.AddLast(Self)
	End Method
	'---------------------------------------------------------------------------------------------------------------------------------
	
	Method Delete()
		If Not _customLuaState
			lua_close(_luaState)
		End If
	End Method
	'---------------------------------------------------------------------------------------------------------------------------------

	Method initLua()
		' inject auxiliary functions into current state
		Local code:String = "function print(...) r=~q~q;for i,v in ipairs(arg) do r=r..tostring(v) end;r=r..~q\n~q;ccOutput(r);end;io.write=print"
		luaL_dostring(_luaState, code)

		'registering auxiliary functions
		lua_register(_luaState, "ccOutput", _consOutput)
		lua_register(_luaState, "ccObjGet", _consGet)
		lua_register(_luaState, "ccObjSet", _consSet)
		lua_register(_luaState, "ccObjDump", _consDump)
		lua_register(_luaState, "ccObjCreate", _consCreate)
		lua_register(_luaState, "ccObjDelete", _consDelete)
		lua_register(_luaState, "ccObjList", _consList)
		lua_register(_luaState, "ccObjToTable", _consObjToTable)
		lua_register(_luaState, "ccClear", _consClear)

		'set debug.traceback as default debugger
		lua_getfield(_luaState, LUA_GLOBALSINDEX, "debug")
		lua_getfield(_luaState, - 1, "traceback")
		lua_remove(_luaState, - 2)

		Local result:Int = lua_pcall(_luaState, 0, LUA_MULTRET, -1)
		If result <> 0
			Local err:String = lua_tostring(_luaState, - 1)
			Throw err
		End If
	End Method
	'---------------------------------------------------------------------------------------------------------------------------------

	Rem
		bbdoc: Sets a custom lua state, e.g. when you use lua yourself in your application.
		returns: -
		about: Sets a custom lua state, e.g. when you use lua yourself in your application. ChaosConsole will inject its functions and terminate the old state automatically.
	End Rem
	Method setLuaState(ls:Byte Ptr)
		If _luaState <> Null
			lua_close(_luaState)
		End If
		
		_luaState = ls
		_customLuaState = True
		initLua()
	End Method
	'---------------------------------------------------------------------------------------------------------------------------------

	Rem
		bbdoc: Register an object for use with the console
		returns: -
		about:
	End Rem
	Method registerObject(name:String, obj:Object)
		_objWrapper.register(name, obj)
		output("Registered object '" + name +"'")
	End Method
	'---------------------------------------------------------------------------------------------------------------------------------
	
	Rem
		bbdoc: Output a line
		returns: -
		about:
	End Rem
	Method output(line:String)
		
		Local tmpLine:String
		
		If line.length > _charsPerLine
			tmpLine = line[_charsPerLine..]
			line = line[.._charsPerLine]
		End If
		
		_lineBuffer = _lineBuffer[1..(_lineBuffer.length+1)]
		_lineBuffer[_lineBuffer.length-1] = line
		
		If tmpLine output(tmpLine)

	End Method
	'---------------------------------------------------------------------------------------------------------------------------------
	
	Rem
		bbdoc: Set console's dimensions
		returns: -
		about: Set width and height of the console
	End Rem
	Method setDimensions(w:Int, h:Int)
		_w = w
		_h = h
		_lineDisplayCount = (_h / _fontHeight) - 1
		_charsPerLine = (_w / TextWidth("M")) - 1
	End Method
	'---------------------------------------------------------------------------------------------------------------------------------
	
	Method toggle()
		_show=1-_show
		If Not _show
			_fadeDir = -1
		Else
			_fadeDir = 1
			_y = -_h
		End If
		FlushKeys()
	End Method
	'---------------------------------------------------------------------------------------------------------------------------------
	
	Rem
		bbdoc: Updates the console
		returns: True, if the console is active
		about: Call this in your main loop. @key defines a key that should be used to toggle the console.
	End Rem
	Method update:Int(key:Int)
		If KeyHit(key) toggle()
		
		If _fadeDir <> 0
			If MilliSecs()-_fadeTimer => 10
				_y:+_fadeDir*10
				_fadeTimer = MilliSecs()
				If _y >= 0 And _fadeDir = 1 
					_show = 1
					_y = 0
					_fadeDir = 0
				End If
				
				If _y <= -_h And _fadeDir = -1
					_show = 0
					_y = -_h
					_fadeDir = 0
				End If
			End If
		End If

		If _show Or _fadeDir
			draw()
			updateInput()
			
			If KeyHit(KEY_PAGEUP) 
				_startLine:+3
				If _startLine+_lineDisplayCount >= _lineBuffer.length
					_startLine = _lineBuffer.length - _lineDisplayCount
				End If
			End If
			
			If KeyHit(KEY_PAGEDOWN) 
				_startLine:-3
				If _startLine <= 0
					_startLine = 0
				End If
			End If
			
			If DelayedKeyDown(DKEY_LEFT)
				If _cursorPos > 0 _cursorPos:-1
			End If
			
			If DelayedKeyDown(DKEY_RIGHT)
				If _cursorPos < _inputLine.length _cursorPos:+1
			End If
			
			If KeyDown(KEY_END)
				_cursorPos = _inputLine.length
			End If
			
			If KeyDown(KEY_HOME)
				_cursorPos = 0
			End If	
			
			If DelayedKeyDown(DKEY_UP) And _historyIndex > 0
				_historyIndex:-1
				_inputLine = _inputHistory[_historyIndex]
				_cursorPos = _inputLine.length
			End If
			
			If DelayedKeyDown(DKEY_DOWN)
				If _historyIndex < _inputHistory.length - 1
					_historyIndex:+1
					_inputLine = _inputHistory[_historyIndex]
					_cursorPos = _inputLine.length
				Else
					If _historyIndex < _inputHistory.length _historyIndex:+1
					_inputLine = ""
					_cursorPos = 0
				End If
			End If
			
			Return True	
		End If
		
	End Method
	'---------------------------------------------------------------------------------------------------------------------------------
	
	Rem
		bbdoc: Set the background color
		returns: -
		about:
	End Rem
	Method setBackgroundColor(r:Byte, g:Byte, b:Byte, a:Float)
		_colorBR = r
		_colorBG = g
		_colorBB = b
		_alpha = a
	End Method
	'---------------------------------------------------------------------------------------------------------------------------------
	
	Rem
		bbdoc: Set an image to draw as console's background
		returns: -
		about:
	End Rem
	Method setBackgroundImage(img:TImage)
		_backImage = img
	End Method
	
	Rem
		bbdoc: Set a color for the font
		returns: -
		about:
	End Rem
	Method setFontColor(r:Byte, g:Byte, b:Byte)
		_colorFR = r
		_colorFG = g
		_colorFB = b
	End Method
	'---------------------------------------------------------------------------------------------------------------------------------
	
	Rem
		bbdoc: Sets a different font for display
		returns: -
		about:
	End Rem
	Method setFont(imgFont:TImageFont)
		_font = imgFont
		Local oldFont:TImageFont = GetImageFont()
		SetImageFont(_font)
		_fontHeight = TextHeight("M")
		setDimensions(_w, _h)
		SetImageFont(oldFont)
	End Method
	'---------------------------------------------------------------------------------------------------------------------------------

	Method draw()
	
		Local oldAlpha:Float = GetAlpha()
		Local oldR:Int, oldG:Int, oldB:Int
		Local oldBlend:Byte = GetBlend()
		Local oldFont:TImageFont = GetImageFont()

		SetImageFont(_font)
		
		GetColor(oldR, oldG, oldB)
	
		SetBlend ALPHABLEND
		SetAlpha _alpha
		SetColor _colorBR, _colorBG, _colorBB
		
		If _backImage 
			DrawImageRect _backImage, 0, _y, _w, _h
		Else
			DrawRect 0, _y, _w, _h
		End If
		
		SetAlpha oldAlpha
		
		SetColor _colorFR, _colorFG, _colorFB
		Local line:Int = _lineDisplayCount+1
		For Local i:Int = _lineBuffer.length-_lineDisplayCount-_startLine Until _lineBuffer.length-_startLine
			DrawText _lineBuffer[i], 5, _h-(line*TextHeight(_lineBuffer[i]))+_y
			line:-1
		Next
		
		If _cursorPos < _inputLine.length
			Local p1:String = _inputLine[.._cursorPos]
			Local p2:String = _inputLine[_cursorPos..]
			DrawText p1, 5, _h-_fontHeight +_y
			DrawText p2, 5+TextWidth("M")*p1.length, _h-_fontHeight +_y
		Else
			DrawText _inputLine, 5, _h-_fontHeight +_y
		End If
		
		DrawLine 0, _h-_fontHeight+_y, _w, _h-_fontHeight+_y
		If MilliSecs()-_cursorBlinkTimer => 300
			_cursorBlinkTimer = MilliSecs()
			_cursorShow = 1 - _cursorShow
		End If
		If _cursorShow DrawRect TextWidth("M")*_cursorPos+5, _h-_fontHeight +_y, 2, _fontHeight-1
		
		SetColor oldR, oldG, oldB
		SetBlend oldBlend
		SetImageFont(oldfont)
	End Method
	'---------------------------------------------------------------------------------------------------------------------------------
	
	Method updateInput()
		Local c:Int = GetChar()
		' normal characters
		If c >= 32 And c <= 125
			If _cursorPos < _inputLine.length
				Local p1:String = _inputLine[.._cursorPos] + Chr(c)
				_inputLine = p1 + _inputLine[_cursorPos..]
			Else
				_inputLine:+Chr(c)
			End If
			_cursorPos:+1
		End If
		
		' backspace
		If c = 8 And _inputLine.length > 0
			If _cursorPos < _inputLine.length
				Local part:String = _inputLine[.._cursorPos-1]
				_inputLine = part + _inputLine[_cursorPos..]
			Else
				_inputLine = _inputLine[.._inputLine.length-1]
			End If
			_cursorPos:-1
		End If
		
		' delete
		If DelayedKeyDown(DKEY_DEL) And _inputLine.length > 0 And _cursorPos < _inputLine.length
			Local part:String = _inputLine[.._cursorPos]
			_inputLine = part + _inputLine[_cursorPos+1..]
		End If
		
		' return
		If c = 13 
			output("> " + _inputLine.Trim())
			parseInput(_inputLine.Trim())
			addHistoryLine(_inputLine)
			_inputLine = ""
			_cursorPos = 0
		End If
	End Method
	'---------------------------------------------------------------------------------------------------------------------------------
	
	Method addHistoryLine(line:String)
		If line.Trim() = "" Return
		
		_historyIndex = _inputHistory.length + 1
		
		If _inputHistory.length > 0
			If line = _inputHistory[_inputHistory.length-1]
				Return
			End If
		End If
		
		_inputHistory = _inputHistory[.._inputHistory.length+1]
		_inputHistory[_inputHistory.length-1] = line
		
	End Method
	'---------------------------------------------------------------------------------------------------------------------------------
	
	Method parseInput:Int(str:String)
		If str = "" Return True
		
		luaL_loadstring(_luaState, str)
		
		Local result:Int = lua_pcall(_luaState, 0, LUA_MULTRET, 0)
		If result <> 0
			Local err:String = lua_tostring(_luaState, - 1)
			'lua_close(_state)
			output("Error: " + err)
		End If

		Return True
	End Method
	'---------------------------------------------------------------------------------------------------------------------------------
	
	Rem
		bbdoc: Register a function to use with the console
		returns: -
		about: @argCount defines how many arguments the function takes at least
	End Rem
	Method registerFunction(name:String, argCount:Int, func:Int(ls:Byte Ptr))
		_funcMap.Insert(name.ToLower(), TFunctionWrapper.Create(name, argCount, func) )
		lua_register(_luaState, name, func)
	End Method
	'---------------------------------------------------------------------------------------------------------------------------------
	
	Method delayedKeyDown:Int(key:Int)
		Local keyc:Int
		Select key
			Case DKEY_LEFT keyc = KEY_LEFT
			Case DKEY_RIGHT keyc = KEY_RIGHT
			Case DKEY_UP keyc = KEY_UP
			Case DKEY_DOWN keyc = KEY_DOWN
			Case DKEY_DEL keyc = KEY_DELETE
		End Select
		
		If MilliSecs()-_keyRepeatTime[key] => 150 And KeyDown(keyc)
			_keyRepeatTime[key] = MilliSecs()
			Return True
		End If
		Return False
	End Method
	'---------------------------------------------------------------------------------------------------------------------------------
	
	Rem
		bbdoc: Clear all text from the console
		returns: -
		about:
	End Rem
	Method clear()
		_lineBuffer = New String[_lineBuffer.length]
	End Method
	'---------------------------------------------------------------------------------------------------------------------------------

End Type
'---------------------------------------------------------------------------------------------------------------------------------